"""
Copyright <2025> <Tuomo Lehtonen, Aalto University>

Permission is hereby granted, free of charge, to any person obtaining a copy of this software and associated documentation files (the "Software"), to deal in the Software without restriction, including without limitation the rights to use, copy, modify, merge, publish, distribute, sublicense, and/or sell copies of the Software, and to permit persons to whom the Software is furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in all copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
"""

import random
import argparse
import os, sys
from collections import defaultdict, deque

def create_clustered_abaf(n_clusters, n_atoms_per_cluster, n_assumptions_per_cluster, max_rules_per_head, max_body_size, attack_prob=0.5):
    """
    Create random ABAF with a clustered structure.
    Parameters:
        n_clusters: number of clusters
        n_atoms_per_cluster: number of non-assumption atoms in each cluster
        n_assumptions_per_cluster: number of assumptions in each cluster
    """

    atom_to_idx = dict()
    index = 1

    assumptions = list()
    atoms = list()
    rules = list()
    rules_deriving = defaultdict(list)

    contraries = {f"asmp_{i}_c_{cluster}" : f"cont_{i}_c_{cluster}" for i in range(1,n_assumptions_per_cluster+1) for cluster in range(1,n_clusters+1)}

    rule_index = 0
    for cluster in range(1,n_clusters+1):
        current_asmpts = [f"asmp_{i}_c_{cluster}" for i in range(1,n_assumptions_per_cluster+1)]
        assumptions.extend(current_asmpts)
        current_atoms = [f"atom_{i}_c_{cluster}" for i in range(1,n_atoms_per_cluster+1)]
        atoms.extend(current_atoms)

        selection_set = current_atoms + current_asmpts

        for atom in current_atoms:
            n_rules_for_atom = random.randint(1, max_rules_per_head)
            for _ in range(n_rules_for_atom):
                body_size = random.randint(1, max_body_size)
                body = random.sample(selection_set, min(body_size, len(selection_set)))

                rules.append((atom, body))
                rules_deriving[atom].append(rule_index)
                rule_index += 1

        # attacks to other cluster
        for c in range(1, n_clusters+1):
            if c == cluster: continue
            for asmp in range(1, n_assumptions_per_cluster+1):
                if random.random() < attack_prob:
                    attacker = random.choice(current_atoms)
                    contrary = f"cont_{asmp}_c_{c}"
                    rules.append((contrary, [attacker]))
                    rules_deriving[contrary].append(rule_index)
                    rule_index += 1

    # creating indices for iccma format
    for a in assumptions:
        atom_to_idx[a] = index
        index += 1
    for a in atoms:
        atom_to_idx[a] = index
        index += 1
    for a,c in contraries.items():
        atom_to_idx[c] = index
        index += 1

    return assumptions, atoms, contraries, rules, rules_deriving, atom_to_idx

class SCCDet:
    def __init__(self, sentences, rules, rules_deriving):
        self.sentences = sentences
        self.rules_deriving = rules_deriving
        self.rules = rules

        self.SCCs = set()
        self.S = deque()
        self.on_stack = set()
        self.i = 0
        self.index = dict()
        self.low = dict()

    def tarjan(self):
        sys.setrecursionlimit(10000)
        for s in self.sentences:
            if not s in self.index:
                self.scc(s)

        return self.SCCs

    def scc(self, s):
        self.index[s] = self.i
        self.low[s] = self.i
        self.i+=1
        self.S.append(s)
        self.on_stack.add(s)
        rules_deriving_s = self.rules_deriving.get(s)
        if rules_deriving_s: # s is derived, so not assumption
            for r in rules_deriving_s:
                for child in self.rules[r][1]:  # body of rth rule
                    if not child in self.index:
                        self.scc(child)
                        self.low[s] = min(self.low[s], self.low[child])
                    elif child in self.on_stack:
                        self.low[s] = min(self.low[s], self.index[child])

        if self.low[s] == self.index[s]:
            SCC = set()
            while True:
                v = self.S.pop()
                self.on_stack.remove(v)
                SCC.add(v)
                if v == s:
                    break

            self.SCCs.add(frozenset(SCC))

def print_ICCMA_format_indexed(assumptions, contraries, rules, n_sentences, out_filename, scc_sizes, atom_to_idx):
    """
    Print the given framework in the ICCMA 2023 format.
    """
    asmpt_set = set(assumptions)
    with open(out_filename, 'w') as out:
        out.write(f"p aba {n_sentences}\n")
        for asm in assumptions:
            asm_index = atom_to_idx.get(asm)
            out.write(f"a {asm_index}\n")
        for asm in contraries.keys():
            asm_index = atom_to_idx.get(asm)
            ctr_index = atom_to_idx.get(contraries.get(asm))
            out.write(f"c {asm_index} {ctr_index}\n")
        for rule in rules:
            head_index = atom_to_idx.get(rule[0])
            body_indices = [atom_to_idx.get(body) for body in rule[1]]
            out.write(f"r {head_index} {' '.join(map(str, body_indices))}\n")
        if scc_sizes:
            out.write(f"# distinct SCC sizes: {scc_sizes}\n")

parser = argparse.ArgumentParser()
parser.add_argument('-d', '--directory', required=True)
parser.add_argument('-i', '--identifier', required=True)
parser.add_argument('-ra', '--rules_per_atom', type=int, default=3)
parser.add_argument('-rs', '--rule_size', type=int, default=2)
parser.add_argument('-ctr', '--ctr_probability', type=int, default=0.8)
parser.add_argument('-c', '--n_clusters', type=int, default=20)
parser.add_argument('-atm', '--n_atoms', type=int, default=20)
parser.add_argument('-asm', '--n_assumptions', type=int, default=20)
parser.add_argument('-n', '--n_instances', type=int, default=5)
parser.add_argument('--log_sccs', action=argparse.BooleanOptionalAction)
parser.add_argument('--seed', type=int)
args = parser.parse_args()

directory = args.directory
identifier = args.identifier

if not os.path.isdir(directory):
    sys.exit(f"Can't find directory: {directory}")

max_rpa = args.rules_per_atom
max_spb = args.rule_size
contrary_prob = args.ctr_probability
n_atoms = args.n_atoms
n_asmps = args.n_assumptions
n_clusters = args.n_clusters

seed = random.randint(1, sys.maxsize)
if args.seed:
    seed = args.seed

file_prefix = f"{identifier}_c{n_clusters}_atoms{n_atoms}_asms{n_asmps}_mra{max_rpa}_mbs{max_spb}_cp{contrary_prob}"
if os.path.isfile(os.path.join(directory, f"{file_prefix}_ins1.aba")):
    sys.exit(f"Files already exist with prefix: {os.path.join(directory, file_prefix)}")

# Generate ABAFs
print(f"Generating ABAFs...")
for i in range(1,args.n_instances+1):
    filename = os.path.join(directory, f"{file_prefix}_ins{i}.aba")

    # if os.path.isfile(filename):
        # sys.exit(f"File already exists: {filename}")

    print(filename)
    framework = create_clustered_abaf(n_clusters, n_atoms, n_asmps, max_rpa, max_spb, contrary_prob)

    scc_sizes = None
    if args.log_sccs:
        # SCC detection; not mandatory
        SCCD = SCCDet(framework[1], framework[3], framework[4])
        sccs = SCCD.tarjan()
        scc_sizes = ({len(scc) for scc in sccs})

    print_ICCMA_format_indexed(framework[0], framework[2], framework[3], n_clusters*(n_atoms+n_asmps*2), filename, scc_sizes, framework[5])

# Write to description file
desc_file = os.path.join(directory, "description.txt")
print(f"Writing to description file: {desc_file}")
with open(desc_file, 'a') as desc:
    desc.write(f"Prefix: {file_prefix}\n")
    desc.write(f"Random seed: {seed}\n")
    desc.write("Parameters:\n")
    desc.write(f" max_rpa = {max_rpa}\n max_spb = {max_spb} \n contrary_prob = {contrary_prob}\n n_atoms = {n_atoms}\n n_asmps = {n_asmps}\n n_clusters = {n_clusters}\n\n")

print(f"Done.")
